package dev.honegger.hackvent2021.santacryptovault.controllers

import dev.honegger.hackvent2021.santacryptovault.services.WalletService
import dev.honegger.hackvent2021.santacryptovault.services.VaultCode
import dev.honegger.hackvent2021.santacryptovault.services.VaultService
import kotlinx.coroutines.*
import mu.KotlinLogging
import org.springframework.http.HttpStatus
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.RestController
import java.util.concurrent.atomic.AtomicInteger
import kotlin.time.Duration.Companion.seconds

/**
 * Prevent evil DDOS or Brute-Force attacks
 */
private const val maxConcurrentRequests = 2

/**
 * Prevent time based Brute-Force attacks
 */
private val constRequestDuration = 2.seconds

private val log = KotlinLogging.logger {  }

@RestController
class VaultController(private val vaultService: VaultService, private val walletService: WalletService) {
    private var activeRequests = AtomicInteger(0)
    private val scope = CoroutineScope(Dispatchers.Default)

    @GetMapping("/check")
    suspend fun check(code: VaultCode): ResponseEntity<String> {
        return if (activeRequests.incrementAndGet() <= maxConcurrentRequests) {
            try {
                log.info { "Checking $code" }
                val delayTask = scope.async { delay(constRequestDuration) }
                val codeTask = scope.async { vaultService.checkCode(code) }

                val res = codeTask.await()
                delayTask.await()
                if (res) {
                    ResponseEntity.ok("Correct code! Here's your crypto wallet: ${walletService.walletAddress}")
                } else {
                    ResponseEntity.status(HttpStatus.FORBIDDEN).body("Wrong code!")
                }
            } finally {
                activeRequests.decrementAndGet()
            }
        } else {
            activeRequests.decrementAndGet()
            log.info { "Blocked DDOS attack" }
            ResponseEntity.status(HttpStatus.TOO_MANY_REQUESTS).body("Too many parallel requests!")
        }
    }
}
